﻿//OVEN
//------------------------------------------------------------------------------
//2014, Lorenzo Buosi, www.chrometaphore.com

// Licence: GPL <http://www.gnu.org/licenses/gpl.html>
//------------------------------------------------------------------------------
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//------------------------------------------------------------------------------

#target photoshop

var _version = "1.0.6";

//prints console msg (debug)
var _verbose = true;

//Save and set units settings into this object
var userPrefs;

//Ink settings. (view init() function)
var settings;

//active document
var doc;

//temp vars in here
var _temp;

//Oven folder
var ovenFolder;
var ovenFolderName = "__oven";

//selected layers IDs array
var selectedIDs;

//current layer into the selected layers loop
var theLayer;
var theLayerID;

var fextRegex = /\.(gif|jpg|jpeg|png)$/i;

//prints console msg
function OvenConsole( msg )
{
	if ( _verbose ) {
		$.writeln( msg );	
	}
}

//display Photoshop Oven alerts 
function OvenAlert( msg ) 
{
	alert( "Oven: " + msg );
}

function initOven( os, documentResolution, outputResolutions, outputFileType, duplicateLayers )
{
	OvenConsole( "Init Oven vars..");

	doc                          = {};
	_temp                        = {};
	ovenFolder                   = {};
	selectedIDs                  = {};
	userPrefs                    = {};
	settings                     = {};
	settings.system              = os;
	//'on' or 'off'
	settings.duplicateLayers     = duplicateLayers;
	settings.exportFile          = {};
	settings.exportFile.exportTo = [];
	//".png" | ".jpg"
	settings.exportFile.type     = outputFileType;

	var myOutput = [];

	switch (settings.system) {
		case "android":
			//only mdpi is supported.
			if ( documentResolution == "M" ) {
				myOutput.push( { resolution:"ldpi", ratio:"75%" } );
				myOutput.push( { resolution:"mdpi", ratio:"100%" } );
				myOutput.push( { resolution:"hdpi", ratio:"150%" } );
				myOutput.push( { resolution:"xhdpi", ratio:"200%" } );
				myOutput.push( { resolution:"xxhdpi", ratio:"300%" } );
			}
		break;
		case "iOS":
			//only @1x and @2x
			if ( documentResolution == "1x" ) {
				myOutput.push( { resolution:"", ratio:"100%" } );
				myOutput.push( { resolution:"@2x", ratio:"200%" } );
				myOutput.push( { resolution:"@3x", ratio:"300%" } );
			}
			else if ( documentResolution == "2x" ) {
				myOutput.push( { resolution:"", ratio:"50%" } );
				myOutput.push( { resolution:"@2x", ratio:"100%" } );
				myOutput.push( { resolution:"@3x", ratio:"150%" } );
			}	
		break;
		default:
			OvenAlert( "Os not recognized. Either 'android' or 'iOS'" );
		break;
	}
	//assign active output resolutions
	for ( var i = 0; i < outputResolutions.length; i++ ) {
		settings.exportFile.exportTo.push( myOutput[outputResolutions[i]] );
	}
 
	theLayer             = {};
	theLayerID           = -1;
}

function Oven( os, documentResolution, outputResolutions, outputFileType, duplicateLayers )
{
	//from string to array
	var myOutputResolutions = outputResolutions.toString().split(",");

	//init vars
	initOven( os, documentResolution, myOutputResolutions, outputFileType, duplicateLayers );

	//let's get reference of currently active PS document
	try
	{
		doc = app.activeDocument;
	} 
	catch ( e )
	{
		OvenAlert( "Oops! No open Photoshop document." );
		return;
	}

	//let's get reference of all selected layer ids before doing anything else.
	selectedIDs = getSelectedLayersIdx();

	//if Oven folder doesnt exist, let's create it.
    try {
    	ovenFolder = doc.layerSets.getByName( ovenFolderName );	
    }
    catch(e) {
    	//..only if duplicate layers is on.
    	if ( settings.duplicateLayers == "on" ) {
    		ovenFolder      = doc.layerSets.add(); 
			ovenFolder.name = ovenFolderName;
		}	
    }

	if ( selectedIDs.length <= 0 )
	{
		OvenAlert( 'No layers selected' );
		return;	
	}

	for( var i = 0; i < selectedIDs.length; i++ ) 
	{
		//select next layer from the ids array
		selectLayerByID( selectedIDs[i] );

		//set reference to currently selected layer
		theLayer   = doc.activeLayer;
		theLayerID = selectedIDs[i];

		var generatorStr  = "";
		var layerOrigName = theLayer.name;
		for ( var k = 0; k < settings.exportFile.exportTo.length; k++ )
		{
			if ( settings.system == "android" ) 
			{
				if ( ! fextRegex.test( layerOrigName ) )
				{
					generatorStr += settings.exportFile.exportTo[k].ratio + " " + settings.exportFile.exportTo[k].resolution + "." + layerOrigName +  settings.exportFile.type;
				}
				else
				{
					generatorStr += settings.exportFile.exportTo[k].ratio + " " + settings.exportFile.exportTo[k].resolution + "." + layerOrigName;	
				}
			} 
			else 
			{
				if ( ! fextRegex.test( layerOrigName ) )
				{
					generatorStr += settings.exportFile.exportTo[k].ratio + " " + layerOrigName + settings.exportFile.exportTo[k].resolution + settings.exportFile.type;	
				}
				else
				{
					generatorStr += settings.exportFile.exportTo[k].ratio + " " + layerOrigName.slice(0, -4) + settings.exportFile.exportTo[k].resolution + settings.exportFile.type;
				}
			}

			/*
			//check if the layer already has a valid file extension
			if ( ! fextRegex.test( layerOrigName ) )
			{
				generatorStr += settings.exportFile.type;
			}
			*/
			
			if ( k != ( settings.exportFile.exportTo.length - 1 ) )
			{
				generatorStr += ", ";	
			}
		}

		if ( settings.duplicateLayers == "on" )
		{
			var duplicate = doc.activeLayer.duplicate( doc, ElementPlacement.INSIDE );
			duplicate.move( ovenFolder, ElementPlacement.INSIDE );
			duplicate.name = generatorStr;
		}
		else
		{
			theLayer.name = generatorStr;
		}
	}
}

function getSelectedLayersIdx() {
	var selectedLayers = new Array;
	var ref = new ActionReference();
	ref.putEnumerated( charIDToTypeID('Dcmn'), charIDToTypeID('Ordn'), charIDToTypeID('Trgt') );
	var desc = executeActionGet(ref);
	if( desc.hasKey( stringIDToTypeID( 'targetLayers' ) ) ){
		desc = desc.getList( stringIDToTypeID( 'targetLayers' ));
		var c = desc.count
		var selectedLayers = new Array();
		for(var i=0;i<c;i++){
			try{
				activeDocument.backgroundLayer;
				selectedLayers.push(  desc.getReference( i ).getIndex() );
			} catch(e){
			selectedLayers.push(  desc.getReference( i ).getIndex()+1 );
		}
	}
	} else {
		var ref = new ActionReference();
		ref.putProperty( charIDToTypeID('Prpr') , charIDToTypeID( 'ItmI' ));
		ref.putEnumerated( charIDToTypeID('Lyr '), charIDToTypeID('Ordn'), charIDToTypeID('Trgt') );
		try{
			activeDocument.backgroundLayer;
			selectedLayers.push( executeActionGet(ref).getInteger(charIDToTypeID( 'ItmI' ))-1);
		} catch(e){
			selectedLayers.push( executeActionGet(ref).getInteger(charIDToTypeID( 'ItmI' )));
		}
	}
	return selectedLayers;
};

function selectLayerByID( index, add )
{
	add == undefined ? add = false : add = true;
	var ref = new ActionReference();
	ref.putIndex(charIDToTypeID("Lyr "), index );
	var desc = new ActionDescriptor();
	desc.putReference(charIDToTypeID( "null" ), ref );
	if ( add ) desc.putEnumerated( stringIDToTypeID( "selectionModifier" ), stringIDToTypeID( "selectionModifierType" ), stringIDToTypeID( "selection" ) );
	desc.putBoolean( charIDToTypeID( "MkVs" ), false );
	try
	{
		executeAction( charIDToTypeID( "slct" ), desc, DialogModes.NO );
	}
	catch(e)
	{
		OvenConsole( e.message );   
	}
};

//Photoshop CC
$._ext_OVEN = {
    run : function( args ) 
    {
    	//os - document resolution - output resolutions
    	var OvenUiArguments = args.split(";");
    	
		Oven( OvenUiArguments[0], OvenUiArguments[1], OvenUiArguments[2], OvenUiArguments[3], OvenUiArguments[4] );	
   
        var appName;	    
	    appName = "Oven";	    
        
        return appName;
    },
};

//test
//$._ext_OVEN.run('android;M;0,1,2,3,4;.png;on');
//$._ext_OVEN.run('iOS;2x;0,1,2;.png;on')